/* 
 *   BuildNet.c
 *
 * Example use of Netica-C API to construct the classic 'Chest Clinic' Bayes net
 * and save it to file 'ChestClinic.dne'.
 *
 * Copyright (C) 1992-2018 by Norsys Software Corp.
 * The software in this file may be copied, modified, and/or included in 
 * derivative works without charge or obligation.
 */

#include <stdio.h>
#include <stdlib.h>
#include "Netica.h"
#include "NeticaEx.h"

#ifndef WDIR
#define WDIR "Data Files/"
#endif

environ_ns* env;


int main (void){
	net_bn* net = NULL;
	node_bn *VisitAsia, *Tuberculosis, *Smoking, *Cancer, *TbOrCa, *XRay, *Bronchitis, *Dyspnea;
	char mesg[MESG_LEN_ns];
	int res;
	
	env = NewNeticaEnviron_ns (NULL, NULL, NULL);
	res = InitNetica2_bn (env, mesg);
	printf ("%s\n", mesg);
	if (res < 0)  return -1;
	
	net = NewNet_bn ("ChestClinic", env);
	CHK_ERR

	VisitAsia =    NewNode_bn ("VisitAsia", 2, net);
	Tuberculosis = NewNode_bn ("Tuberculosis", 2, net);
	Smoking =      NewNode_bn ("Smoking", 2, net);
	Cancer =       NewNode_bn ("Cancer", 2, net);
	TbOrCa =       NewNode_bn ("TbOrCa", 2, net);
	XRay =         NewNode_bn ("XRay", 2, net);
	Dyspnea =      NewNode_bn ("Dyspnea", 2, net);
	Bronchitis =   NewNode_bn ("Bronchitis", 2, net);
	CHK_ERR

	SetNodeStateNames_bn (VisitAsia,   "visit,   no_visit");
	SetNodeStateNames_bn (Tuberculosis,"present, absent");
	SetNodeStateNames_bn (Smoking,     "smoker,  nonsmoker");
	SetNodeStateNames_bn (Cancer,      "present, absent");
	SetNodeStateNames_bn (TbOrCa,      "true,    false");
	SetNodeStateNames_bn (XRay,        "abnormal,normal");
	SetNodeStateNames_bn (Dyspnea,     "present, absent");
	SetNodeStateNames_bn (Bronchitis,  "present, absent");
	SetNodeTitle_bn (TbOrCa, "Tuberculosis or Cancer");
	SetNodeTitle_bn (Cancer, "Lung Cancer");
	CHK_ERR

	AddLink_bn (VisitAsia, Tuberculosis);
	AddLink_bn (Smoking, Cancer);
	AddLink_bn (Tuberculosis, TbOrCa);
	AddLink_bn (Cancer, TbOrCa);
	AddLink_bn (TbOrCa, XRay);
	AddLink_bn (TbOrCa, Dyspnea);
	AddLink_bn (Smoking, Bronchitis);
	AddLink_bn (Bronchitis, Dyspnea);
	CHK_ERR
	
	// WARNING: doubles must be passed to SetNodeProbs, eg, 0.0 not 0

	SetNodeProbs (VisitAsia, 0.01, 0.99);
	
	SetNodeProbs (Tuberculosis, "visit",    0.05, 0.95);
	SetNodeProbs (Tuberculosis, "no_visit", 0.01, 0.99);

	SetNodeProbs (Smoking, 0.5, 0.5);

	SetNodeProbs (Cancer, "smoker",    0.1,  0.9);
	SetNodeProbs (Cancer, "nonsmoker", 0.01, 0.99);

	SetNodeProbs (Bronchitis, "smoker",    0.6, 0.4);
	SetNodeProbs (Bronchitis, "nonsmoker", 0.3, 0.7);

	//                   Tuberculosis Cancer
	SetNodeEquation_bn (TbOrCa, "TbOrCa (Tuberculosis, Cancer) = Tuberculosis==present || Cancer==present");
	EquationToTable_bn (TbOrCa, 1, 0, 0);

	//                  TbOrCa  Abnormal Normal
	SetNodeProbs (XRay, "true",  0.98,  0.02);
	SetNodeProbs (XRay, "false", 0.05,  0.95);

	//                     TbOrCa   Bronchitis 
	SetNodeProbs (Dyspnea, "true",  "present", 0.9, 0.1);
	SetNodeProbs (Dyspnea, "true",  "absent",  0.7, 0.3);
	SetNodeProbs (Dyspnea, "false", "present", 0.8, 0.2);
	SetNodeProbs (Dyspnea, "false", "absent",  0.1, 0.9);
	CHK_ERR

	WriteNet_bn (net,  NewFileStream_ns (WDIR"ChestClinic.dne", env, NULL));
	CHK_ERR

	printf ("Created file 'ChestClinic.dne' in the '%s' directory.\n", WDIR);

	DeleteNet_bn (net);
	res = CloseNetica_bn (env, mesg);

	printf ("%s\n", mesg);
	printf ("Press <enter> key to quit\n");
	getchar();
	return (res < 0 ? -3 : 0);
}

