/* 
 *   LearnCPTs.c
 *
 * Example use of Netica-C API for learning the conditional probability tables
 * (CPTs) of a Bayes net from a file of cases.
 *
 * It reads in Bayes net 'ChestClinic.dne', built by the BuildNet.c example,
 * removes its CPTs, and then learns new ones from the casefile 'ChestClinic.cas'
 * (built by SimulateCases example).
 * It then outputs the final net to file 'Learned_ChestClinic.dne'.
 *
 * Copyright (C) 1992-2018 by Norsys Software Corp.
 * The software in this file may be copied, modified, and/or included in 
 * derivative works without charge or obligation.
 * This file contains example software only, and Norsys makes no warranty that 
 * it is suitable for any particular purpose, or without defects.
 */

#include <stdio.h>
#include <stdlib.h>
#include "Netica.h"
#include "NeticaEx.h"

#ifndef WDIR
#define WDIR "Data Files/"
#endif

environ_ns* env;


int main (void){
	net_bn *orig_net = NULL, *learned_net = NULL;
	const nodelist_bn* orig_nodes;
	nodelist_bn* learned_nodes = NULL;
	int numnodes;
	stream_ns* casefile;
	char mesg[MESG_LEN_ns];
	int i, res;
	
	env = NewNeticaEnviron_ns (NULL, NULL, NULL);
	res = InitNetica2_bn (env, mesg);
	printf ("%s\n", mesg);
	if (res < 0)  return -1;
	
	// Read in the net created by the BuildNet.c example program
	orig_net = ReadNet_bn (NewFileStream_ns (WDIR"ChestClinic.dne", env, NULL), NO_VISUAL_INFO);
	orig_nodes = GetNetNodes2_bn (orig_net, NULL);
	SetNetAutoUpdate_bn (orig_net, 0);
	CHK_ERR

	learned_net = NewNet_bn ("Learned_ChestClinic", env);
	learned_nodes = CopyNodes_bn (orig_nodes, learned_net, NULL);
	numnodes = LengthNodeList_bn (learned_nodes);

	// Remove CPTables of nodes in learned_net, so new ones can be learned.
	for (i = 0;  i < numnodes;  ++i)
		DeleteNodeTables_bn (NthNode_bn (learned_nodes, i));
	CHK_ERR
	
	// Read in the case file created by the the SimulateCases.c
	//   example program, and learn new CPTables.
	casefile = NewFileStream_ns (WDIR"ChestClinic.cas", env, NULL);
	ReviseCPTsByCaseFile_bn (casefile, learned_nodes, 0, 1.0);

	WriteNet_bn (learned_net,  NewFileStream_ns (WDIR"Learned_ChestClinic.dne", env, NULL));
	CHK_ERR

	printf ("\nLearned new CPTs for the ChestClinic Bayes net, \nusing the data file: '%sChestClinic.cas'.\n", WDIR);
	printf ("Put the result as file 'Learned_ChestClinic.dne' in the '%s' directory.\n", WDIR);
	
	DeleteNodeList_bn (learned_nodes);
	DeleteNet_bn (orig_net);
	DeleteNet_bn (learned_net);
	res= CloseNetica_bn (env, mesg);
	printf ("%s\n", mesg);
	printf ("Press <enter> key to quit\n", mesg);
	getchar();
	return (res < 0 ? -3 : 0);
}

	/* ==============================================================
	 * This alternate way can replace the ReviseCPTsByCaseFile_bn 
	 * line above, if you need to filter or adjust individual cases.

	case_posn = FIRST_CASE;
	while(1){
		RetractNetFindings_bn (learned_net);
		ReadCase_bn (&case_posn, casefile, learned_nodes, NULL, NULL);
		if (case_posn == NO_MORE_CASES)  break;
		ReviseCPTsByFindings_bn (learned_nodes, 0, 1.0);
		case_posn = NEXT_CASE;
		CHK_ERR
		}

	  ============================================================== */
