/* 
 *   MakeDecision.c
 *
 * Example use of Netica-C API to build the classic 'Umbrella' decision net 
 * and use it to find the expected utility of each decision choice and the 
 * optimal decision table, given different conditions.
 *
 * Copyright (C) 1992-2018 by Norsys Software Corp.
 * The software in this file may be copied, modified, and/or included in 
 * derivative works without charge or obligation.
 * This file contains example software only, and Norsys makes no warranty that 
 * it is suitable for any particular purpose, or without defects.
 */

#include <stdio.h>
#include <stdlib.h>
#include "Netica.h"
#include "NeticaEx.h"

#ifndef WDIR
#define WDIR "Data Files/"
#endif

environ_ns* env;


int main (void){
	net_bn* net = NULL;
	node_bn *weather, *forecast, *umbrella, *satisfaction;
	state_bn fs, decision;
	const util_bn* utils;
	char mesg[MESG_LEN_ns];
	int res;

	env = NewNeticaEnviron_ns (NULL, NULL, NULL);
	res = InitNetica2_bn (env, mesg);
	printf ("%s\n", mesg);
	if (res < 0)  return -1;
	
	net = NewNet_bn ("Umbrella", env);
	CHK_ERR

	weather  = NewNode_bn ("Weather",  2, net);
	forecast = NewNode_bn ("Forecast", 3, net);
	umbrella = NewNode_bn ("Umbrella", 2, net);
	SetNodeKind_bn (umbrella, DECISION_NODE);
	satisfaction = NewNode_bn ("Satisfaction", 0, net);
	SetNodeKind_bn (satisfaction, UTILITY_NODE);
	CHK_ERR

	SetNodeStateNames_bn (forecast,"sunny,        cloudy,   rainy");
	SetNodeStateNames_bn (weather, "sunshine,     rain");
	SetNodeStateNames_bn (umbrella,"take_umbrella,dont_take_umbrella");
	CHK_ERR

	AddLink_bn (weather,  forecast);
	AddLink_bn (forecast, umbrella);
	AddLink_bn (weather,  satisfaction);
	AddLink_bn (umbrella, satisfaction);
	CHK_ERR
	
	SetNodeProbs (weather, 0.7, 0.3);
	
	//                                  forecast
	//                       weather |  sunny  cloudy rainy
	SetNodeProbs (forecast, "sunshine", 0.7,   0.2,   0.1);
	SetNodeProbs (forecast, "rain",     0.15,  0.25,  0.6);

	//                                   weather     umbrella
	SetNodeFuncReal (satisfaction,  20, "sunshine", "take_umbrella");
	SetNodeFuncReal (satisfaction, 100, "sunshine", "dont_take_umbrella");
	SetNodeFuncReal (satisfaction,  70, "rain",     "take_umbrella");
	SetNodeFuncReal (satisfaction,   0, "rain",     "dont_take_umbrella");
	CHK_ERR

	CompileNet_bn (net);
	
	//--- 1st type of usage:  To get the expected utilities, given the current findings

	EnterFinding ("Forecast", "sunny", net);
	utils = GetNodeExpectedUtils2_bn (umbrella, NULL);  // returns expected utilities, given current findings

	printf ("If the forecast is sunny, expected utility of %s is %f, of %s is %f\n",
		GetNodeStateName_bn (umbrella, 0), utils[0],
		GetNodeStateName_bn (umbrella, 1), utils[1]);
	CHK_ERR
	
	RetractNetFindings_bn (net);
	EnterFinding ("Forecast", "cloudy", net);
	utils = GetNodeExpectedUtils2_bn (umbrella, NULL);

	printf ("If the forecast is cloudy, expected utility of %s is %f, of %s is %f\n\n",
		GetNodeStateName_bn (umbrella, 0), utils[0],
		GetNodeStateName_bn (umbrella, 1), utils[1]);
	CHK_ERR

	//--- 2nd type of usage:  To get the optimal decision table

	RetractNetFindings_bn (net);
	GetNodeExpectedUtils2_bn (umbrella, NULL);   // causes Netica to recompute decision tables, given current findings
	                                             // (which in this case are no findings)

	for (fs = 0;  fs < GetNodeNumberStates_bn (forecast);  ++fs){
		decision = GetNodeFuncState_bn (umbrella, &fs);
		printf ("If the forecast is '%s', the best decision is %s.\n",
		        GetNodeStateName_bn (forecast, fs),
		        GetNodeStateName_bn (umbrella, decision));
	}
	CHK_ERR

	DeleteNet_bn (net);
	res= CloseNetica_bn (env, mesg);
	printf ("%s\n", mesg);
	printf ("Press <enter> key to quit\n", mesg);
	getchar();
	return (res < 0 ? -3 : 0);
}
