/* 
 * NeticaCPP.cpp
 *
 * Source file for C++ Development with Netica API, version 5.04 and higher.
 *
 * See file NeticaCPP.h for the header file.
 * For more information, see www.norsys.com
 *
 * Copyright (C) 2000-2019 by Norsys Software Corp.
 *
 * This file may be freely used, copied, modified and distributed in source or binary form.
 * No warranty is provided for it, and Norsys will not be liable for any damages or losses resulting from its use.
 * This does not provide any license rights to software not in this file.
 */

#include <climits>
#include <cstdio>
#include <cstdlib>

#pragma warning (disable: 4996)		// using the regular version of Standard Library, instead of Microsoft's secure one, because we need to be portable

#include "NeticaCPP.h"

namespace netica {


Environ* GlobalEnv = 0;


Environ::Environ (const char license[], Environ* env, const char locn[]){
	START_ns
	cenv = NewNeticaEnviron_ns (license, env ? env->cenv : 0, locn);
	cenvloc = env ? env->cenv : cenv;
	if (GlobalEnv == 0)  GlobalEnv = this;
	SetEnvironUserData_ns (cenv, 0, this);
	SetLanguage_ns (cenv, "C++");
	char mesg[MESG_LEN_ns];
	int ret = InitNetica2_bn (cenv, mesg);
	if (ret < 0) throw NeticaError (static_cast<Environ*>(NULL), mesg, 5, ERROR_ERR);
	END_ns
	}

/*____________________________________________________________________________ StrLen
*/
static int StrLen (const char* str){
	const char* sp = str;
	while (*sp) ++sp;
	size_t len = sp - str;
	if (len > INT_MAX)  len = INT_MAX;		// truncate any strings longer than INT_MAX (may want to generate error mesg instead)
	return static_cast<int>(len);			// cast ok, because of above truncate
	}

/*______________________________________________________________ User::getNthFieldName
'buf' must have room for at least NAME_MAX_ns characters.
*/
const char* User::getNthFieldName (int index){
	const char* name = NULL;
	START_ns
	if      (what == FOR_NET)  GetNetNthUserField_bn  (getNet().getCPtr(),  index, &name, NULL, NULL, 0);
	else if (what == FOR_NODE) GetNodeNthUserField_bn (getNode().getCPtr(), index, &name, NULL, NULL, 0);
	END2_ns
	return name;
}

/*_________________________________________________________________ User::removeField
*/
void User::removeField (const char field_name[]){
	START_ns
	if      (what == FOR_NET)  SetNetUserField_bn  (getNet().getCPtr(),  field_name, NULL, 0, 0);
	else if (what == FOR_NODE) SetNodeUserField_bn (getNode().getCPtr(), field_name, NULL, 0, 0);
	END2_ns
}

/*__________________________________________________________________ User::setString
*/
void User::setString (const char field_name[], const char str[]){
	START_ns
	if      (what == FOR_NET)  SetNetUserField_bn  (getNet().getCPtr(),  field_name, str, StrLen (str), 0);
	else if (what == FOR_NODE) SetNodeUserField_bn (getNode().getCPtr(), field_name, str, StrLen (str), 0);
	END2_ns
}

/*__________________________________________________________________ User::getString
*/
const char* User::getString (const char field_name[]){
	START_ns
	if      (what == FOR_NET)  return GetNetUserField_bn  (getNet().getCPtr(),  field_name, NULL, 0);
	else if (what == FOR_NODE) return GetNodeUserField_bn (getNode().getCPtr(), field_name, NULL, 0);
	END2_ns
	return NULL;
}

/*___________________________________________________________________ User::setBytes
*/
void User::setBytes (const char field_name[], const char bytes[], int num_bytes){
	START_ns
	if      (what == FOR_NET)  SetNetUserField_bn  (getNet().getCPtr(),  field_name, bytes, num_bytes, 0);
	else if (what == FOR_NODE) SetNodeUserField_bn (getNode().getCPtr(), field_name, bytes, num_bytes, 0);
	END2_ns
}

/*___________________________________________________________________ User::getBytes
On return, num_bytes will contain the number of bytes.
*/
const char* User::getBytes (const char field_name[], int& num_bytes){
	START_ns
	if      (what == FOR_NET)  return GetNetUserField_bn  (getNet().getCPtr(),  field_name, &num_bytes, 0);
	else if (what == FOR_NODE) return GetNodeUserField_bn (getNode().getCPtr(), field_name, &num_bytes, 0);
	END2_ns
	return NULL;
}

/*__________________________________________________________________ User::setNumber
*/
void User::setNumber (const char field_name[], double field_value){
	START_ns
	char buf[65];
	sprintf (buf, "%g", field_value);					// The below is better, but if your system doesn't have it, use this instead
	// snprintf (buf, sizeof(buf), "%g", field_value);
	if      (what == FOR_NET)  SetNetUserField_bn  (getNet().getCPtr(),  field_name, buf, StrLen (buf), 0);
	else if (what == FOR_NODE) SetNodeUserField_bn (getNode().getCPtr(), field_name, buf, StrLen (buf), 0);
	END2_ns
}

/*__________________________________________________________________ User::getNumber
*/
double User::getNumber (const char field_name[]){
	START_ns
	int length = -1;
	const char* str = NULL;
	if      (what == FOR_NET)   str = GetNetUserField_bn  (getNet().getCPtr(),  field_name, &length, 0);
	else if (what == FOR_NODE)  str = GetNodeUserField_bn (getNode().getCPtr(), field_name, &length, 0);
	if (length == -1)  NewError_ns (Environ::getDefault()->getCPtr(), 955, ERROR_ERR, "There is no user field with that name");
	else {
		char* end;
		double num = strtod (str, &end);
		if (*end != 0)  NewError_ns (Environ::getDefault()->getCPtr(), 956, ERROR_ERR, "The user field did not hold a valid number");
		else return num;
	}
	END2_ns
	return 0;
}

/*____________________________________________________________________ User::setInteger
*/
void User::setInteger (const char field_name[], int field_value){
	START_ns
	char buf[22];
	sprintf (buf, "%d", field_value);					// The below is better, but if your system doesn't have it, use this instead
	// snprintf (buf, sizeof(buf), "%d", field_value);
	if      (what == FOR_NET)  SetNetUserField_bn  (getNet().getCPtr(),  field_name, buf, StrLen (buf), 0);
	else if (what == FOR_NODE) SetNodeUserField_bn (getNode().getCPtr(), field_name, buf, StrLen (buf), 0);
	END2_ns
}

/*____________________________________________________________________ User::getInteger
*/
int User::getInteger (const char field_name[]){
	START_ns
	int length = -1;
	const char* str = NULL;
	if      (what == FOR_NET)   str = GetNetUserField_bn  (getNet().getCPtr(),  field_name, &length, 0);
	else if (what == FOR_NODE)  str = GetNodeUserField_bn (getNode().getCPtr(), field_name, &length, 0);
	if (length == -1)  NewError_ns (Environ::getDefault()->getCPtr(), 953, ERROR_ERR, "There is no user field with that name");
	else {
		char* end;
		long num = strtol (str, &end, 10);
		if (*end != 0)  NewError_ns (Environ::getDefault()->getCPtr(), 954, ERROR_ERR, "The user field did not hold a valid integer");
		else return num;
	}
	END2_ns
	return 0;
}

/*____________________________________________________________________________ NewError
Like NewError_ns, but with printf style arguments for the error message.
*/
#include <stdarg.h>

report_ns* NewError (int number, errseverity_ns severity, const char* mesg, ...){
	va_list ap;
	char buf[450];
	va_start (ap, mesg);
	// vsprintf (buf, mesg, ap);					// The below is better, but if your system doesn't have it, use this instead (but then be careful mesg is not too long)
	vsnprintf (buf, sizeof(buf)-1, mesg, ap);  buf[sizeof(buf)-1] = 0;		// should work with Standard definition of vsnprintf and Microsoft's one
	va_end (ap);
	return  NewError_ns (Environ::getDefault()->getCPtr(), number, severity, buf);
}



}  // namespace netica



