
/* 
 * NeticaDemo.c
 *
 * Example use of Netica API Programmers Module, version 1.04.
 *
 * This is a combination of the first 2 examples in the Netica API
 * Reference Manual.  It can be used to test if the installation is successful.
 *
 * Copyright (C) 1992-1996 by Norsys Software Corp.
 * The software in this file may be copied, modified, and/or included in 
 * derivative works without charge or obligation.
 * Norsys makes no guarantee of continued support for the software in this file.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include "Netica.h"
#include "NeticaEx.h"

#define CHKERR  {if (GetError_ns (env, ERROR_ERR, NULL))  goto error;}

extern environ_ns *env;

void main (void){
	net_bn *net;
	node_bn *VisitAsia, *Tuberculosis, *Smoking, *Cancer, *TbOrCa, *XRay;
	double belief;
	char mesg[MESG_LEN_ns];
	int res;
	
	printf ("\nWelcome to Netica API!\n");
	printf ("This demo project is from the first 2 examples of the Reference Manual.\n");
	printf ("If it compiles, links and runs successfully (i.e. without generating\n");
	printf ("errors), then your installation is probably good.\n");
	printf ("Now you can replace the Demo.c file with your own code.\n");
	printf ("--------------------------------------------------------------------\n\n");
	
	env = NewNeticaEnviron_bn (NULL);
	res = InitNetica_bn (&env, mesg);

	printf ("%s\n", mesg);
	if (res < 0)  exit (-1);
	
	net = NewNet_bn ("AsiaEx", env);
	CHKERR

	VisitAsia =    NewNode_bn ("VisitAsia", 2, net);
	Tuberculosis = NewNode_bn ("Tuberculosis", 2, net);
	Smoking =      NewNode_bn ("Smoking", 2, net);
	Cancer =       NewNode_bn ("Cancer", 2, net);
	TbOrCa =       NewNode_bn ("TbOrCa", 2, net);
	XRay =         NewNode_bn ("XRay", 2, net);
	CHKERR

	SetNodeStateNames (VisitAsia,   "Visit",   "No_Visit");
	SetNodeStateNames (Tuberculosis,"Present", "Absent");
	SetNodeStateNames (Smoking,     "Smoker",  "NonSmoker");
	SetNodeStateNames (Cancer,      "Present", "Absent");
	SetNodeStateNames (TbOrCa,      "True",    "False");
	SetNodeStateNames (XRay,        "Abnormal","Normal");
	SetNodeTitle_bn (TbOrCa, "Tuberculosis or Cancer");
	SetNodeTitle_bn (Cancer, "Lung Cancer");
	CHKERR

	AddLink_bn (VisitAsia, Tuberculosis);
	AddLink_bn (Smoking, Cancer);
	AddLink_bn (Tuberculosis, TbOrCa);
	AddLink_bn (Cancer, TbOrCa);
	AddLink_bn (TbOrCa, XRay);
	CHKERR
	
	SetNodeProbs (VisitAsia, 0.01, 0.99);
	
	SetNodeProbs (Tuberculosis, "Visit",    0.05, 0.95);
	SetNodeProbs (Tuberculosis, "No_Visit", 0.01, 0.99);

	SetNodeProbs (Smoking, 0.5, 0.5);

	SetNodeProbs (Cancer, "Smoker",    0.1,  0.9);
	SetNodeProbs (Cancer, "NonSmoker", 0.01, 0.99);

	//                   Tuberculosis Cancer
	SetNodeProbs (TbOrCa, "Present", "Present", 1.0, 0.0);
	SetNodeProbs (TbOrCa, "Present", "Absent",  1.0, 0.0);
	SetNodeProbs (TbOrCa, "Absent",  "Present", 1.0, 0.0);
	SetNodeProbs (TbOrCa, "Absent",  "Absent",  0.0, 1.0);

	//                  TbOrCa  Abnormal Normal
	SetNodeProbs (XRay, "True",  0.98,  0.02);
	SetNodeProbs (XRay, "False", 0.05,  0.95);
	CHKERR

	CompileNet_bn (net);
	CHKERR

	belief = GetNodeBelief ("Tuberculosis", "Present", net);
	CHKERR

	printf ("The probability of tuberculosis is %g\n\n", belief);

	EnterFinding ("XRay", "Abnormal", net);
	belief = GetNodeBelief ("Tuberculosis", "Present", net);
	CHKERR

	printf ("Given an abnormal X-ray, \n\
         the probability of tuberculosis is %g\n\n", belief);

	EnterFinding ("VisitAsia", "Visit", net);
	belief = GetNodeBelief ("Tuberculosis", "Present", net);
	CHKERR

	printf ("Given an abnormal X-ray and a visit to Asia, \n\
	     the probability of tuberculosis is %g\n\n", belief);

	EnterFinding ("Cancer", "Present", net);
	belief = GetNodeBelief ("Tuberculosis", "Present", net);
	CHKERR

	printf ("Given abnormal X-ray, Asia visit, and lung cancer, \n\
         the probability of tuberculosis is %g\n\n", belief);

end:
	FreeNet_bn (net);
	res = CloseNetica_bn (env, mesg);
	printf ("%s\n", mesg);
	printf ("Press <enter> key to quit ", mesg);
	getchar();
	exit (res < 0 ? -1 : 0);

error:
	fprintf (stderr, "NeticaDemo: Error in %s\n", 
	         ErrorMessage_ns (GetError_ns (env, ERROR_ERR, NULL)));
	goto end;
}

