/**
 * An extension of norsys.netica.Net containing assorted convenience methods.
 * Users are welcome to cut-and-paste methods from this class into their
 * own extensions of norsys.netica.Net.
 *   WARNING: The methods in this class could change, so do not rely on them
 * always being here in future versions of Netica-J.  They might be promoted 
 * to the base class, revised, or removed entirely.
 */

import norsys.netica.Node;
import norsys.netica.Net;
import norsys.netica.*;
import java.util.*;

public class NetEx extends Net{

  //------------------------------------------------------------ constructors
  public NetEx() throws NeticaException {
      super();
  }

  public NetEx( Environ env ) throws NeticaException { 
      super (env);
  }

  public NetEx( Streamer inStream ) throws NeticaException { 
      super (inStream);
  }

  //------------------------------------------------------------
    

  /** 
   *  Make a copy of a net, giving it the name, newName.
   *  @param newName  the name of the new net.
   */
  public static Net duplicateNet(Net net, String newName) throws NeticaException {
    Net newNet = new Net (net.getEnviron());
    newNet.doc().setName (newName);
    NodeList newNodes = net.duplicateNodes (net.getNodes(), newNet);
       
    NodeList elimOrder = net.getElimOrder();
    NodeList newOrder =NodeListEx.mapNodeList (elimOrder, newNet);
    newNet.setElimOrder (newOrder);

    newNet.setAutoUpdate (net.getAutoUpdate());
    newNet.doc().setTitle (net.doc().getTitle());
    newNet.doc().setComment (net.doc().getComment());
    newNet.doc().setUserData (net.doc().getUserData());           // If desired
    return newNet;
  }

 
  /** 
   *  A convenience routine based on Node.enterFinding( String stateName ).
   *  @param the name of the node in this net whose finding you wish to set.
   *  @param stateName the name of the state that this node is believed to be in.
   */
  public void enterFinding( String nodeName, String stateName ) throws NeticaException {
      Node node = getNode ( nodeName);
      node.enterFinding (stateName);
  }

  /** 
   *  Set a user field to an ascii string.  
   *  If the string is not ascii, throw an exception.
   *  @param fieldname the userField name; must be IDname or fewer characters.
   *  @param asciiStr  the userField value that is supposed to be an ascii string.
   */
  public void setUserAsciiString (String fieldname, String asciiStr ) throws NeticaException {
      if ( ! Util.isAscii( asciiStr ) ) {
	  throw new NeticaException( 20000, "Not an ascii string" );
      }
     doc().setUserField (fieldname, asciiStr);
  }

  /** 
   *  Set a user field to a double written as a string.
   *  @param fieldname the userField name; must be IDname or fewer characters.
   *  @param num       a double whose String representation will be set to the userField's value,
   */
  public void setUserNumber (String fieldname, double num) throws NeticaException {
      doc().setUserField (fieldname, ""+num);
  }

  /** 
   *  Returns a user field and confirms it is an ascii string.  
   *  If the string is not ascii, throw an exception.
   *  @param fieldname the userField name; must be IDname or fewer characters.
   *  @returns the userField value that is supposed to be an ascii string.
   *  @throws NeticaException if the 
   */
  public String getUserAsciiString (String fieldname ) throws NeticaException {
      String asciiStr = (String) doc().getUserField (fieldname);
      if ( ! Util.isAscii (asciiStr)) {
	  throw new NeticaException (20000, "Not an ascii string");
      }
      return asciiStr;
  }

  /** 
   *  Retrieves a number that was written to a user field with setUserNumber().
   *  @param fieldname the userField name; must be IDname or fewer characters.
   *  @returns the userField value that is supposed to be a number.
   */
  public double getUserNumber (String fieldname) throws Exception {
      String value = (String) doc().getUserField (fieldname);
      return Double.parseDouble (value);
  }

  /** 
   *  Find a net in the given Environ by it's name. Comparison must be an exact match and 
   *  is case sensitive.  Returns the first net that matches, or null, if there was no match.
   *  @param  name The name of the net sought.
   *  @param  env  The Environ to search within.
   *  @returns The net with that name, or null, if not found.
   */
  public static Net getNetNamed (String name, Environ env) throws NeticaException { 
      Vector nets = Net.getAllNets( env );
      Enumeration enum = nets.elements();
      while (enum.hasMoreElements()) {
	  Net net = (Net) enum.nextElement();
	  if (net.doc().getName().equals (name)) {
	      return net;
	  }
      }
      return null;
  }

  /** 
   *  Find a node by name.  Like getNode(String nodeName), except that
   *  throws a NeticaException if the name doesn't exist.
   *  @param  name The name of the node sought.
   *  @returns The node with that name.
   */
  public Node getExistingNode (String nodeName) throws NeticaException{
      Node node = getNode (nodeName);
      if (node == null){
	  throw new NeticaException (20000, //any number between 20000 and 29999
				     "There is no node named " + nodeName +
				     " in net " + doc().getName());
      }
      return node;
  }
}

